<?php
// dashboard.php

session_start();

// Check if admin is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

// Include the database connection
require 'db_connection.php';

// Handle form submission for adding consultation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_consultation'])) {
    $consultation_name = mysqli_real_escape_string($conn, $_POST['consultation_name']);
    $consultation_fee = mysqli_real_escape_string($conn, $_POST['consultation_fee']);
    $consultation_details = mysqli_real_escape_string($conn, $_POST['consultation_details']);

    if (!empty($consultation_name) && !empty($consultation_fee) && !empty($consultation_details)) {
        $sql = "INSERT INTO consultations (consultation_name, consultation_fee, consultation_details) VALUES (?, ?, ?)";
        if ($stmt = $conn->prepare($sql)) {
            $stmt->bind_param("sds", $consultation_name, $consultation_fee, $consultation_details);
            if ($stmt->execute()) {
                $message = "Consultation added successfully.";
            } else {
                $message = "Error adding consultation: " . $stmt->error;
            }
            $stmt->close();
        } else {
            $message = "Error preparing statement: " . $conn->error;
        }
    } else {
        $message = "All fields are required.";
    }
}

// Handle delete consultation
if (isset($_GET['delete_id'])) {
    $delete_id = (int)$_GET['delete_id'];
    $sql = "DELETE FROM consultations WHERE id = ?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $delete_id);
        if ($stmt->execute()) {
            $message = "Consultation deleted successfully.";
        } else {
            $message = "Error deleting consultation: " . $stmt->error;
        }
        $stmt->close();
    }
}

// Fetch consultations for view/edit
$consultations = [];
$sql = "SELECT * FROM consultations";
$result = $conn->query($sql);
if ($result && $result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        $consultations[] = $row;
    }
}

// Handle edit consultation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['edit_consultation'])) {
    $consultation_id = (int)$_POST['consultation_id'];
    $consultation_name = mysqli_real_escape_string($conn, $_POST['consultation_name']);
    $consultation_fee = mysqli_real_escape_string($conn, $_POST['consultation_fee']);
    $consultation_details = mysqli_real_escape_string($conn, $_POST['consultation_details']);

    if (!empty($consultation_name) && !empty($consultation_fee) && !empty($consultation_details)) {
        $sql = "UPDATE consultations SET consultation_name = ?, consultation_fee = ?, consultation_details = ? WHERE id = ?";
        if ($stmt = $conn->prepare($sql)) {
            $stmt->bind_param("sdsi", $consultation_name, $consultation_fee, $consultation_details, $consultation_id);
            if ($stmt->execute()) {
                $message = "Consultation updated successfully.";
            } else {
                $message = "Error updating consultation: " . $stmt->error;
            }
            $stmt->close();
        } else {
            $message = "Error preparing statement: " . $conn->error;
        }
    } else {
        $message = "All fields are required.";
    }
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Consultations</title>
    <link rel="stylesheet" href="styles.css">
    <style>/* General Styles */
body {
    font-family: Arial, sans-serif;
    margin: 0;
    padding: 0;
    background-color: #f9f9f9;
    color: #333;
}

.container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
}

h1, h2 {
    text-align: center;
    margin-bottom: 20px;
}

p {
    text-align: center;
    font-size: 1rem;
    margin-bottom: 20px;
}

/* Form Styles */
form {
    background-color: #fff;
    padding: 20px;
    border-radius: 8px;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
    margin-bottom: 20px;
}

form div {
    margin-bottom: 15px;
}

label {
    display: block;
    font-weight: bold;
    margin-bottom: 5px;
}

input[type="text"],
input[type="number"],
textarea {
    width: 100%;
    padding: 10px;
    border: 1px solid #ccc;
    border-radius: 4px;
    font-size: 1rem;
}

button {
    background-color: #007bff;
    color: #fff;
    border: none;
    padding: 10px 20px;
    font-size: 1rem;
    border-radius: 4px;
    cursor: pointer;
}

button:hover {
    background-color: #0056b3;
}

/* Table Styles */
table {
    width: 100%;
    border-collapse: collapse;
    margin-bottom: 20px;
}

table th,
table td {
    padding: 10px;
    border: 1px solid #ccc;
    text-align: left;
}

table th {
    background-color: #007bff;
    color: #fff;
}

a {
    color: #007bff;
    text-decoration: none;
}

a:hover {
    text-decoration: underline;
}

/* Responsive Design */
@media (max-width: 768px) {
    .container {
        padding: 10px;
    }

    h1, h2 {
        font-size: 1.5rem;
    }

    form, table {
        font-size: 0.9rem;
    }

    button {
        width: 100%;
        padding: 10px;
    }
}

@media (min-width: 769px) {
    form {
        max-width: 600px;
        margin: 0 auto;
    }
}
</style>
</head>
<body>
    <div class="container">
        <h1>Manage Consultations</h1>
        <?php if (isset($message)): ?>
            <p><?php echo $message; ?></p>
        <?php endif; ?>

        <!-- Add Consultation Form -->
        <form action="" method="POST">
            <input type="hidden" name="add_consultation">
            <h2>Add Consultation</h2>
            <div>
                <label for="consultation_name">Consultation Name:</label>
                <input type="text" id="consultation_name" name="consultation_name" required>
            </div>
            <div>
                <label for="consultation_fee">Consultation Fee:</label>
                <input type="number" id="consultation_fee" name="consultation_fee" step="0.01" required>
            </div>
            <div>
                <label for="consultation_details">Consultation Details:</label>
                <textarea id="consultation_details" name="consultation_details" rows="5" required></textarea>
            </div>
            <button type="submit">Add Consultation</button>
        </form>

        <!-- Consultations Table -->
        <h2>Consultations</h2>
        <table border="1">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Name</th>
                    <th>Fee</th>
                    <th>Details</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($consultations as $consultation): ?>
                    <tr>
                        <td><?php echo $consultation['id']; ?></td>
                        <td><?php echo htmlspecialchars($consultation['consultation_name']); ?></td>
                        <td><?php echo htmlspecialchars($consultation['consultation_fee']); ?></td>
                        <td><?php echo htmlspecialchars($consultation['consultation_details']); ?></td>
                        <td>
                            <form action="" method="POST" style="display:inline;">
                                <input type="hidden" name="edit_consultation">
                                <input type="hidden" name="consultation_id" value="<?php echo $consultation['id']; ?>">
                                <input type="text" name="consultation_name" value="<?php echo htmlspecialchars($consultation['consultation_name']); ?>" required>
                                <input type="number" name="consultation_fee" step="0.01" value="<?php echo htmlspecialchars($consultation['consultation_fee']); ?>" required>
                                <textarea name="consultation_details" rows="3" required><?php echo htmlspecialchars($consultation['consultation_details']); ?></textarea>
                                <button type="submit">Edit</button>
                            </form>
                            <a href="?delete_id=<?php echo $consultation['id']; ?>" onclick="return confirm('Are you sure you want to delete this consultation?');">Delete</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</body>
</html>
