<?php
// Include database connection
require 'db_connection.php';

// Fetch all products and their images
$products = $pdo->query('
    SELECT products.*, GROUP_CONCAT(product_images.image_path) AS images
    FROM products
    LEFT JOIN product_images ON products.id = product_images.product_id
    GROUP BY products.id
')->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html>
<head>
    <title>Product List</title>
    <style>
        /* Basic styling */
        table {
            width: 100%;
            border-collapse: collapse;
        }
        table, th, td {
            border: 1px solid black;
        }
        th, td {
            padding: 10px;
            text-align: left;
        }
        img {
            max-width: 100px;
            height: auto;
        }
        .image-list {
            display: flex;
            gap: 10px;
        }
    </style>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
</head>
<body>
    <h2>Product List</h2>
    <table>
        <thead>
            <tr>
                <th>Product Name</th>
                <th>Old Price</th>
                <th>Current Price</th>
                <th>Discount</th>
                <th>Quantity</th>
                <th>Images</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($products as $product): ?>
                <tr id="product-row-<?= $product['id'] ?>">
                    <td><?= htmlspecialchars($product['product_name']) ?></td>
                    <td><?= htmlspecialchars($product['old_price']) ?></td>
                    <td><?= htmlspecialchars($product['current_price']) ?></td>
                    <td><?= round($product['discount'], 2) ?>%</td>
                    <td><?= htmlspecialchars($product['quantity']) ?></td>
                    <td>
                        <div class="image-list">
                            <?php 
                            $images = explode(',', $product['images']);
                            foreach ($images as $image): 
                            ?>
                                <img src="<?= htmlspecialchars($image) ?>" alt="Product Image">
                            <?php endforeach; ?>
                        </div>
                    </td>
                    <td>
                        <button class="edit-btn" data-id="<?= $product['id'] ?>">Edit</button>
                        <button class="delete-btn" data-id="<?= $product['id'] ?>">Delete</button>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <!-- Modal for Editing -->
    <div id="editModal" style="display:none;">
        <h2>Edit Product</h2>
        <form id="editForm">
            <input type="hidden" name="product_id" id="product_id">
            <label for="product_name">Product Name:</label>
            <input type="text" name="product_name" id="product_name"><br>

            <label for="old_price">Old Price:</label>
            <input type="number" step="0.01" name="old_price" id="old_price"><br>

            <label for="current_price">Current Price:</label>
            <input type="number" step="0.01" name="current_price" id="current_price"><br>

            <label for="quantity">Quantity:</label>
            <input type="number" name="quantity" id="quantity"><br>

            <button type="button" onclick="updateProduct()">Save Changes</button>
            <button type="button" onclick="closeEditModal()">Cancel</button>
        </form>
    </div>

    <script>
        // Edit Button Click Handler
        $('.edit-btn').click(function() {
            var productId = $(this).data('id');
            $.ajax({
                url: 'fetch_product.php',
                type: 'POST',
                data: { id: productId },
                success: function(response) {
                    var product = JSON.parse(response);
                    $('#product_id').val(product.id);
                    $('#product_name').val(product.product_name);
                    $('#old_price').val(product.old_price);
                    $('#current_price').val(product.current_price);
                    $('#quantity').val(product.quantity);
                    $('#editModal').show();
                }
            });
        });

        // Update Product AJAX
        function updateProduct() {
            var formData = $('#editForm').serialize();
            $.ajax({
                url: 'update_product.php',
                type: 'POST',
                data: formData,
                success: function(response) {
                    alert('Product updated successfully');
                    location.reload(); // Refresh page after updating
                }
            });
        }

        // Delete Button Click Handler
        $('.delete-btn').click(function() {
            if (confirm('Are you sure you want to delete this product?')) {
                var productId = $(this).data('id');
                $.ajax({
                    url: 'delete_product.php',
                    type: 'POST',
                    data: { id: productId },
                    success: function(response) {
                        $('#product-row-' + productId).remove();
                        alert('Product deleted successfully');
                    }
                });
            }
        });

        // Close Edit Modal
        function closeEditModal() {
            $('#editModal').hide();
        }
    </script>
</body>
</html>
